<?php

namespace DynamicContentForElementor;

class Features {
	/**
	 * all main features that can be activated/deactivated independently.
	 */
	public $all_features;
	/**
	 * @var array<string,array<string,string>>
	 */
	public $bundled_features;
	private $features_status = false;
	const FEATURES_STATUS_OPTION = 'dce_features_status_option';

	public function __construct() {
		$this->all_features = self::get_all_features_data();
		$status = $this->get_features_status();
		$this->bundled_features = self::get_bundled_features_data();
		$this->update_all_features_with_new_status( $status );
		$this->save_later_if_empty();
	}

	public function save_features() {
		update_option( self::FEATURES_STATUS_OPTION, wp_json_encode( $this->get_features_status() ) );
	}

	/**
	 * Avoid saving immediately to avoid races with db upgrades.
	 *
	 * @return void
	 */
	public function save_later_if_empty() {
		$option_json = get_option( self::FEATURES_STATUS_OPTION );
		if ( $option_json === false ) {
			$hook = 'dce_auto_save_features';
			add_action( $hook, [ $this, 'save_features' ] );
			if ( ! wp_next_scheduled( $hook ) ) {
				wp_schedule_single_event( time() + 15 * 60, $hook );
			}
		}
	}

	/**
	 * Get Widgets Groups
	 *
	 * @return array<string,string>
	 */
	public static function get_widgets_groups() {
		return [
			'LIST' => esc_html__( 'List', 'dynamic-content-for-elementor' ),
			'FAVORITES' => esc_html__( 'Favorites', 'dynamic-content-for-elementor' ),
			'MAPS' => esc_html__( 'Maps', 'dynamic-content-for-elementor' ),
			'PDF' => 'PDF',
			'CONTENT' => esc_html__( 'Content', 'dynamic-content-for-elementor' ),
			'CREATIVE' => esc_html__( 'Creative', 'dynamic-content-for-elementor' ),
			'DEV' => esc_html__( 'Developer', 'dynamic-content-for-elementor' ),
			'WOO' => 'WooCommerce',
			'ACF' => esc_html__( 'Advanced Custom Fields', 'dynamic-content-for-elementor' ),
			'METABOX' => 'Meta Box',
			'TOOLSET' => 'Toolset',
			'PODS' => 'Pods',
			'SVG' => 'SVG',
			'WEBGL' => 'WEBGL',
			'POST' => esc_html__( 'Post', 'dynamic-content-for-elementor' ),
		];
	}

	/**
	 * Get Extensions Groups
	 *
	 * @return array<string,string>
	 */
	public static function get_extensions_groups() {
		return [
			'COMMON' => esc_html__( 'for All Widgets', 'dynamic-content-for-elementor' ),
			'FREE' => esc_html__( 'for Elementor Free', 'dynamic-content-for-elementor' ),
			'FORM' => esc_html__( 'for Elementor Pro Form', 'dynamic-content-for-elementor' ),
		];
	}

	/**
	 * Get Dynamic Tags Groups
	 *
	 * @return array<string,string>
	 */
	public static function get_dynamic_tags_groups() {
		return [
			'POST' => esc_html__( 'Posts and Pages', 'dynamic-content-for-elementor' ),
			'USER' => esc_html__( 'Users', 'dynamic-content-for-elementor' ),
			'WOO' => 'WooCommerce',
			'TERMS' => esc_html__( 'Terms', 'dynamic-content-for-elementor' ),
			'CONTENT' => esc_html__( 'Content', 'dynamic-content-for-elementor' ),
			'DEV' => esc_html__( 'Developer', 'dynamic-content-for-elementor' ),
			'ACF' => esc_html__( 'Advanced Custom Fields', 'dynamic-content-for-elementor' ),
			'METABOX' => 'Meta Box',
			'DYNAMIC_GOOGLE_MAPS_DIRECTIONS' => 'Dynamic Google Maps Directions',
		];
	}

	public function get_feature_info( string $feature_id, string $info ) {
		return $this->all_features[ $feature_id ][ $info ];
	}

	/**
	 * Get Feature Title
	 *
	 * @param string $feature_id
	 * @return string
	 */
	public function get_feature_title( string $feature_id ) {
		return $this->all_features[ $feature_id ]['title'];
	}

	public function get_feature_info_by_array( array $features, string $key ) {
		$return = [];
		foreach ( $features as $feature ) {
			$return[] = $feature[ $key ];
		}
		return $return;
	}

	public function filter( array $args, string $operator = 'AND' ) {
		return wp_list_filter( $this->all_features, $args, $operator );
	}

	/**
	 * Filter Bundled
	 *
	 * @param array<mixed> $args
	 * @param string $operator
	 * @return array<mixed>
	 */
	public function filter_bundled( array $args, string $operator = 'AND' ) {
		return wp_list_filter( $this->bundled_features, $args, $operator );
	}

	public function filter_by_tag( string $value ) {
		return array_filter( $this->all_features, function ( $f ) use ( $value ) {
			return in_array( $value, $f['tag'] ?? [], true );
		} );
	}

	/**
	 * Filter By Collection
	 *
	 * @param string $value
	 * @return array<string,array<string,string>>
	 */
	public function filter_by_collection( string $value ) {
		return array_filter( $this->all_features, function ( $f ) use ( $value ) {
			return isset( $f['collection'] ) && $value === $f['collection'];
		} );
	}

	public function get_active_features() {
		return array_filter( $this->get_features_status(), function ( $e ) {
			return 'active' === $e;
		} );
	}

	/**
	 * Is Feature Active
	 *
	 * @param string $feature
	 * @return boolean
	 */
	public function is_feature_active( string $feature ) {
		return $this->get_feature_info( $feature, 'status' ) === 'active';
	}

	public function update_all_features_with_new_status( $status ) {
		array_walk( $this->all_features, function ( &$v, $k ) use ( $status ) {
			$v['status'] = $status[ $k ];
		});
		// bundled features should have the same status as the main features
		// they are bundled with:
		foreach ( $this->bundled_features as $key => $data ) {
			$status = $this->all_features[ $data['activated_by'] ]['status'];
			$this->bundled_features[ $key ]['status'] = $status;
		}
	}

	/**
	 * Return an associative array with the feature name as key and the value
	 * either 'active' or 'inactive' based on the status of the feature.
	 *
	 * @return array<string,string>
	 */
	public function get_features_status() {
		if ( $this->features_status === false ) {
			$option_json = get_option( self::FEATURES_STATUS_OPTION, false );
			if ( $option_json === false ) {
				// features never saved, use the default status.
				$default = array_map( function ( $info ) {
					if ( ( $info['default_status'] ?? 'active' ) === 'inactive' ) {
						return 'inactive';
					}
					$is_legacy = isset( $info['legacy'] ) && $info['legacy'];
					return $is_legacy ? 'inactive' : 'active';
				}, $this->all_features );
				$this->features_status = $default;
			} else {
				// every feature that is not saved in the db should be inactive.
				$option = json_decode( $option_json, true );
				$default = array_map( function ( $info ) {
					return 'inactive';
				}, $this->all_features );
				$this->features_status = $option + $default;
			}
		}
		return $this->features_status;
	}


	/**
	 * Received an assoc array with key feature name and value feature status.
	 * This can be a subset of all features. Update the feature option in the db with the provided values
	 * leaving unchanged the features not provided.
	 *
	 * @return void
	 */
	public function db_update_features_status( $subset_status ) {
		$status_before = array_map( function ( $f ) {
			return $f['status'];
		}, $this->all_features );
		$new_status = $subset_status + $status_before;
		update_option( self::FEATURES_STATUS_OPTION, wp_json_encode( $new_status ) );
		$this->update_all_features_with_new_status( $new_status );
	}

	/**
	 * Get Bundled Features Data
	 *
	 * @return array<string,mixed>
	 */
	public static function get_bundled_features_data() {
		return [
			'ext_tag_dynamic_google_maps_directions_info' => [
				'activated_by' => 'wdg_google_maps_directions',
				'name' => 'ext_tag_dynamic_google_maps_directions_info',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\DynamicGoogleMapsDirectionsInfo',
				'category' => 'DYNAMIC_GOOGLE_MAPS_DIRECTIONS',
				'extension_type' => 'dynamic-tag',
				'title' => esc_html__( 'Dynamic Tag - Map Info', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Shows information from a map created with Dynamic Google Maps Directions', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-google-maps-directions-info',
				'plugin_depends' => '',
				'doc_url' => 'dynamic-tag-map-info',
			],
			'ext_tag_dynamic_google_maps_directions_instructions' => [
				'activated_by' => 'wdg_google_maps_directions',
				'name' => 'ext_tag_dynamic_google_maps_directions_instructions',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\DynamicGoogleMapsDirectionsInstructions',
				'category' => 'DYNAMIC_GOOGLE_MAPS_DIRECTIONS',
				'extension_type' => 'dynamic-tag',
				'title' => esc_html__( 'Dynamic Tag - Map Instructions', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Shows the road information of a map created with Dynamic Google Maps Directions', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-google-maps-directions-instructions',
				'plugin_depends' => '',
				'doc_url' => 'dynamic-tag-map-instructions',
			],
		];
	}

	/**
	 * Get All Features Data
	 *
	 * @return array<string,mixed>
	 */
	public static function get_all_features_data() {
		return [
			///  Extensions ///
			'ext_animations' => [
				'name' => 'dce_extension_animations',
				'type' => 'extension',
				'class' => 'Extensions\DynamicAnimations',
				'category' => 'COMMON',
				'title' => esc_html__( 'Dynamic Animations', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Predefined CSS-Animations with keyframe', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-animations',
				'plugin_depends' => '',
				'doc_url' => 'loop-animations/',
			],
			'ext_masking' => [
				'name' => 'dce_extension_masking',
				'type' => 'extension',
				'class' => 'Extensions\AdvancedMasking',
				'category' => 'FREE',
				'title' => esc_html__( 'Advanced Masking Controls', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Advanced Masking features for Image, Image-box and Video Widgets', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-advanced-masking',
				'plugin_depends' => '',
				'doc_url' => 'advanced-masking/',
			],
			'ext_rellax' => [
				'name' => 'dce_extension_rellax',
				'type' => 'extension',
				'class' => 'Extensions\Rellax',
				'category' => 'COMMON',
				'title' => esc_html__( 'Rellax', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Rellax Parallax rules for widgets and rows', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-rellax',
				'plugin_depends' => '',
				'doc_url' => 'rellax-parallax/',
			],
			'ext_reveal' => [
				'name' => 'dce_extension_reveal',
				'type' => 'extension',
				'class' => 'Extensions\Reveal',
				'category' => 'COMMON',
				'title' => esc_html__( 'Reveal', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Reveal animation on-scroll for widgets', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-reveal',
				'plugin_depends' => '',
				'doc_url' => 'scroll-reveals/',
			],
			'ext_transforms' => [
				'name' => 'dce_extension_transforms',
				'type' => 'extension',
				'class' => 'Extensions\Transforms',
				'category' => 'COMMON',
				'title' => esc_html__( 'Transforms', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Apply CSS Transforms to Element', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-transforms',
				'plugin_depends' => '',
				'doc_url' => 'transforms/',
			],
			'ext_video' => [
				'name' => 'dce_extension_video',
				'type' => 'extension',
				'class' => 'Extensions\AdvancedVideoControls',
				'category' => 'FREE',
				'title' => esc_html__( 'Advanced Video Controls', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Advanced Video features for Elementor Video Widget', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-advanced-video-controls',
				'plugin_depends' => '',
				'doc_url' => 'advanced-video-controls/',
			],
			'ext_visibility' => [
				'name' => 'dce_extension_visibility',
				'type' => 'extension',
				'class' => 'Extensions\DynamicVisibility',
				'category' => 'COMMON',
				'title' => esc_html__( 'Dynamic Visibility', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Visibility rules for widgets, rows, columns, and sections', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-visibility',
				'plugin_depends' => '',
				'doc_url' => 'dynamic-visibility-for-elementor/',
			],
			'ext_form_address_autocomplete' => [
				'name' => 'dce_extension_form_address_autocomplete',
				'type' => 'extension',
				'class' => 'Extensions\AddressAutocomplete',
				'category' => 'FORM',
				'tag' => [ 'enchanted' ],
				'title' => esc_html__( 'Address Autocomplete for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Use autocomplete to give your form fields the type-ahead-search behaviour of the Google Maps search field. The autocomplete service can match on full words and substrings, resolving place names and addresses', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-autocomplete-address',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'address-autocomplete-for-elementor-pro-form/',
			],
			'ext_form_amount' => [
				'name' => 'dce_extension_form_amount',
				'type' => 'extension',
				'class' => 'Extensions\Amount',
				'category' => 'FORM',
				'title' => esc_html__( 'Amount Field for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add an Amount Field to Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-amount',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'amount-elementor-pro-form/',
			],
			'ext_form_range' => [
				'name' => 'dce_extension_form_range',
				'type' => 'extension',
				'class' => 'Extensions\Range',
				'category' => 'FORM',
				'title' => esc_html__( 'Range Field for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add a Range Field to Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-range',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'range-field-elementor-pro-form/',
			],
			'ext_form_description' => [
				'name' => 'dce_extension_form_description',
				'type' => 'extension',
				'class' => 'Extensions\FieldDescription',
				'category' => 'FORM',
				'tag' => [ 'enchanted' ],
				'title' => esc_html__( 'Field Description for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Describe your form field to help users better understand each field. You can add a tooltip or insert text', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-description',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'field-description-elementor-pro-form/',
			],
			'ext_form_email' => [
				'name' => 'dce_extension_form_email',
				'type' => 'extension',
				'class' => 'Extensions\DynamicEmail',
				'category' => 'FORM',
				'title' => esc_html__( 'Dynamic Email for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add Dynamic Email action to Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-email',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'dynamic-email-for-elementor-pro-form/',
			],
			'ext_form_favorites_action' => [
				'name' => 'dce_extension_form_favorites_action',
				'type' => 'extension',
				'class' => 'Extensions\FavoritesAction',
				'category' => 'FORM',
				'title' => esc_html__( 'Favorites Action for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add favorites via Elementor forms. Useful to use in combination with the "Add to Favorites" and "Favorites" widgets', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-favorites-action',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'favorites-elementor-pro-form/',
			],
			'ext_form_export' => [
				'name' => 'dce_extension_form_export',
				'type' => 'extension',
				'class' => 'Extensions\Export',
				'category' => 'FORM',
				'title' => esc_html__( 'Export for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add Export action to Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-export',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'export-for-elementor-pro-form/',
			],
			'ext_form_icons' => [
				'name' => 'dce_extension_form_icons',
				'type' => 'extension',
				'class' => 'Extensions\Icons',
				'category' => 'FORM',
				'tag' => [ 'enchanted' ],
				'title' => esc_html__( 'Icons for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add icons on the label or inside the input of form fields', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-icons',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'icons-for-elementor-pro-form/',
			],
			'ext_form_inline_align' => [
				'name' => 'dce_extension_form_inline_align',
				'type' => 'extension',
				'class' => 'Extensions\InlineAlign',
				'category' => 'FORM',
				'tag' => [ 'enchanted' ],
				'title' => esc_html__( 'Inline Align for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Choose the inline align type for checkbox and radio fields', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-inline-align',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'inline-align-for-elementor-pro-form/',
			],
			'ext_form_length' => [
				'name' => 'dce_extension_form_length',
				'type' => 'extension',
				'class' => 'Extensions\FieldLength',
				'category' => 'FORM',
				'tag' => [ 'enchanted' ],
				'title' => esc_html__( 'Field Length for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Choose a minimum and maximum characters length for your text and textarea fields', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-field-length',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'field-length-for-elementor-pro-form/',
			],
			'ext_form_message' => [
				'name' => 'dce_extension_form_message',
				'type' => 'extension',
				'class' => 'Extensions\MessageGenerator',
				'category' => 'FORM',
				'title' => esc_html__( 'Message Generator for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Options to customize the Elementor Pro Form success message', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-message-generator',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'message-generator-for-elementor-pro-form/',
			],
			'ext_form_method' => [
				'name' => 'dce_extension_form_method',
				'type' => 'extension',
				'class' => 'Extensions\Method',
				'category' => 'FORM',
				'title' => esc_html__( 'Method for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add a different method attribute on your forms that specifies how to send form-data', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-form-method',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'method-for-elementor-pro-form/',
			],
			'ext_form_password_visibility' => [
				'name' => 'dce_extension_form_password_visibility',
				'type' => 'extension',
				'class' => 'Extensions\PasswordVisibility',
				'category' => 'FORM',
				'tag' => [ 'enchanted' ],
				'title' => esc_html__( 'Password Visibility for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Allow your users to show or hide their password on Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-password-visibility',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'password-visibility-for-elementor-pro-form/',
			],
			'ext_form_unique_id' => [
				'name' => 'dce_form_unique_id',
				'type' => 'extension',
				'class' => 'Extensions\UniqueId',
				'category' => 'FORM',
				'title' => esc_html__( 'Unique ID Field for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'With this hidden field you can generate a unique ID to be used in PDF Generator or Dynamic Email', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-unique-id',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'unique-id-elementor-pro-form/',
			],
			'ext_form_counter' => [
				'name' => 'dce_form_counter',
				'type' => 'extension',
				'class' => 'Extensions\Counter',
				'category' => 'FORM',
				'title' => esc_html__( 'Counter Field for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Perfect for keeping track of submissions or the sum of another form field', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-counter',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'counter-elementor-pro-form/',
			],
			'ext_form_pdf' => [
				'name' => 'dce_extension_form_pdf',
				'type' => 'extension',
				'class' => 'Extensions\PdfGenerator',
				'category' => 'FORM',
				'title' => esc_html__( 'PDF Generator for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add PDF creation action to Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-pdf-generator',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'pdf-generator-for-elementor-pro-form/',
			],
			'ext_form_php_action' => [
				'name' => 'php_action',
				'type' => 'extension',
				'class' => 'Extensions\PhpAction',
				'category' => 'FORM',
				'title' => esc_html__( 'PHP Action for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add PHP code to the actions of a form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-form-php-action',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'php-action-elementor-pro-form/',
			],
			'ext_form_cookie_action' => [
				'name' => 'dce_cookie_action',
				'type' => 'extension',
				'class' => 'Extensions\CookieAction',
				'category' => 'FORM',
				'title' => esc_html__( 'Cookie Action for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Set cookies after submitting the form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-cookie-action',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'cookie-action-elementor-pro-form/',
			],
			'ext_form_redirect' => [
				'name' => 'dce_extension_form_redirect',
				'type' => 'extension',
				'class' => 'Extensions\DynamicRedirect',
				'category' => 'FORM',
				'title' => esc_html__( 'Dynamic Redirect for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Redirect your users to different URLs based on their choice on submitted form fields', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-redirect',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'dynamic-redirect-for-elementor-pro-form/',
			],
			'ext_form_regex' => [
				'name' => 'dce_extension_form_regex',
				'type' => 'extension',
				'class' => 'Extensions\RegexField',
				'category' => 'FORM',
				'tag' => [ 'enchanted' ],
				'title' => esc_html__( 'Regex Field for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Validate form fields using a regular expression', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-regex',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'regex-field-for-elementor-pro-form/',
			],
			'ext_form_reset' => [
				'name' => 'dce_extension_form_reset',
				'type' => 'extension',
				'class' => 'Extensions\ResetButton',
				'category' => 'FORM',
				'title' => esc_html__( 'Reset Button for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add a reset button which resets all form fields to their initial values', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-reset',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'reset-button-for-elementor-pro-form/',
			],
			'ext_form_save' => [
				'name' => 'dce_extension_form_save',
				'type' => 'extension',
				'class' => 'Extensions\Save',
				'category' => 'FORM',
				'title' => esc_html__( 'Save for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Save the form data submitted by your client as posts, users or terms', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-save',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'save-elementor-pro-form/',
			],
			'ext_form_select2' => [
				'name' => 'dce_extension_form_select2',
				'type' => 'extension',
				'class' => 'Extensions\Select2',
				'category' => 'FORM',
				'tag' => [ 'enchanted' ],
				'title' => esc_html__( 'Select2 for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add Select2 to your select fields to gives a customizable select box with support for searching', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-select2',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'select2-for-elementor-pro-form/',
			],
			'ext_form_signature' => [
				'name' => 'dce_extension_form_signature',
				'type' => 'extension',
				'class' => 'Extensions\Signature',
				'category' => 'FORM',
				'title' => esc_html__( 'Signature for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add a signature field to Elementor Pro Form and it will be included in your PDF', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-signature',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'signature-field-for-elementor-pro-form',
			],
			'ext_form_step' => [
				'name' => 'dce_extension_form_step',
				'type' => 'extension',
				'class' => 'Extensions\EnhancedMultiStep',
				'category' => 'FORM',
				'title' => esc_html__( 'Enhanced Multi-Step for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add features to Elementor Pro Multi-Step: label as a legend, show all steps, scroll to top on step change and step summary', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-multistep',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'enhanced-multi-step-elementor-pro-form/',
				'legacy' => true,
			],
			'ext_form_submit_on_change' => [
				'name' => 'dce_extension_form_submit_on_change',
				'type' => 'extension',
				'class' => 'Extensions\SubmitOnChange',
				'category' => 'FORM',
				'tag' => [ 'enchanted' ],
				'title' => esc_html__( 'Submit On Change for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Submit the form automatically when the user chooses a radio button or a select field', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-submit-on-change',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'submit-on-change-for-elementor-pro-form/',
			],
			'ext_form_submit' => [
				'name' => 'dce_extension_form_submit',
				'type' => 'extension',
				'class' => 'Extensions\SubmitButton',
				'category' => 'FORM',
				'title' => esc_html__( 'Submit Button for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add another submit button on your forms', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-submit',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'submit-button-for-elementor-pro-form/',
			],
			'ext_form_telegram' => [
				'name' => 'dce_extension_form_telegram',
				'type' => 'extension',
				'class' => 'Extensions\Telegram',
				'category' => 'FORM',
				'title' => esc_html__( 'Telegram for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Send the content of your Elementor Pro Form to Telegram', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-telegram',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'telegram-for-elementor-pro-form/',
			],
			'ext_form_visibility' => [
				'name' => 'dce_extension_form_visibility',
				'type' => 'extension',
				'class' => 'Extensions\ConditionalFieldsOldVersion',
				'legacy' => true,
				'replaced_by' => 'ext_conditional_fields_v2',
				'category' => 'FORM',
				'title' => esc_html__( 'Conditional Fields (old version) for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Conditionally display fields based on form field values', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-conditional-fields',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'conditional-fields-for-elementor-pro-form/',
			],
			'ext_conditional_fields_v2' => [
				'name' => 'dce_conditional_fields_v2',
				'type' => 'extension',
				'class' => 'Extensions\ConditionalFields',
				'category' => 'FORM',
				'title' => esc_html__( 'Conditionals for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Contains the extensions: Conditional Fields, Conditional Validation, Conditional Mailchimp, Conditional Active Campaign, Conditional Webhook, Conditional MailerLite, and Max Submissions for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-conditional-fields',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'conditional-fields-v2-for-elementor-pro-form/',
			],
			'ext_form_wysiwyg' => [
				'name' => 'ext_form_wysiwyg',
				'type' => 'extension',
				'class' => 'Extensions\WYSIWYG',
				'category' => 'FORM',
				'title' => esc_html__( 'WYSIWYG Field for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add a WYSIWYG Field to your forms', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-wysiwyg',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'wysiwyg-field-elementor-pro-form/',
			],
			'ext_form_paypal' => [
				'name' => 'dce_extension_form_paypal',
				'type' => 'extension',
				'class' => 'Extensions\PaypalField',
				'category' => 'FORM',
				'title' => esc_html__( 'PayPal Field for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add a PayPal field for simple payments to Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-paypal',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'paypal-for-elementor-pro-form/',
			],
			'ext_form_stripe' => [
				'name' => 'dce_extension_form_stripe',
				'type' => 'extension',
				'class' => 'Extensions\StripeField',
				'category' => 'FORM',
				'title' => esc_html__( 'Stripe Field for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add a Stripe field for simple payments to Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-stripe',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'stripe-for-elementor-pro-form/',
			],
			'ext_custom_validation' => [
				'name' => 'dce_custom_validation',
				'type' => 'extension',
				'class' => 'Extensions\PhpValidation',
				'category' => 'FORM',
				'title' => esc_html__( 'PHP Validation for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add PHP code to validate a whole form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-custom-validation',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'php-validation-for-elementor-pro-form/',
			],
			'ext_js_field' => [
				'name' => 'dce_js_field',
				'type' => 'extension',
				'class' => 'Extensions\JsField',
				'category' => 'FORM',
				'title' => esc_html__( 'JS Field for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Take advantage of JS full power to set the value of a field on Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-js-amount',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'js-field-elementor-pro-form/',
			],
			'ext_mirror_field' => [
				'name' => 'dce_mirror_field',
				'type' => 'extension',
				'class' => 'Extensions\MirrorField',
				'category' => 'FORM',
				'title' => esc_html__( 'Mirror Field for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'A dynamic tool that can auto-fill your field values and yet allow user modification when needed', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-mirror-field',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'mirror-field-elementor-pro-form/',
			],
			'ext_formatted_number_field' => [
				'name' => 'dce_formatted_number_field',
				'type' => 'extension',
				'class' => 'Extensions\FormattedNumber',
				'category' => 'FORM',
				'title' => esc_html__( 'Formatted Number for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'A number field that is displayed formatted on blur', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-formatted-number',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'formatted-number-elementor-pro-form/',
			],
			'ext_live_html' => [
				'name' => 'dce_live_html',
				'type' => 'extension',
				'class' => 'Extensions\LiveHtml',
				'category' => 'FORM',
				'title' => esc_html__( 'Live HTML Field for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Insert the values of your fields in an HTML Field and those changes will be updated live as they happen', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-live-html',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'live-html-field-elementor-pro-form/',
			],
			'ext_hidden_label' => [
				'name' => 'dce_hidden_label',
				'type' => 'extension',
				'class' => 'Extensions\HiddenLabel',
				'category' => 'FORM',
				'title' => esc_html__( 'Hidden Label for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Get the label corresponding to the value of another Radio, Select or Checkbox field', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-hidden-label',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'hidden-label-elementor-pro-form/',
			],
			'ext_dynamic_select' => [
				'name' => 'dce_dynamic_select',
				'type' => 'extension',
				'class' => 'Extensions\DynamicSelect',
				'category' => 'FORM',
				'title' => esc_html__( 'Dynamic Select for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Insert a select field where the list of options changes dynamically according to the value of another field', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-select',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'dynamic-select-field-elementor-pro-form/',
			],
			'ext_tooltip' => [
				'name' => 'dce_tooltip',
				'type' => 'extension',
				'class' => 'Extensions\Tooltip',
				'category' => 'COMMON',
				'title' => esc_html__( 'Tooltip', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add a tooltip to any widget', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-tooltip',
				'plugin_depends' => [],
				'doc_url' => 'tooltip-elementor',
			],
			'ext_dynamic_countdown' => [
				'name' => 'dce_dynamic_countdown',
				'type' => 'extension',
				'class' => 'Extensions\DynamicCountdown',
				'category' => 'PRO',
				'legacy' => true,
				'replaced_by_custom_message' => esc_html__( 'This feature has been deprecated since Elementor Pro added new dynamic tag for due date in v3.10.0.', 'dynamic-content-for-elementor' ),
				'title' => esc_html__( 'Dynamic Due Date for Elementor Countdown', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add a Dynamic Due Date field on Elementor Countdown widget to permit dynamic values', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-countdown',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'dynamic-due-date-for-elementor-countdown/',
			],
			'ext_confirm_dialog' => [
				'name' => 'dce_confirm_dialog',
				'type' => 'extension',
				'class' => 'Extensions\ConfirmDialog',
				'category' => 'FORM',
				'title' => esc_html__( 'Confirm Dialog for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'An easy-to-use component that allows you to use Confirm/Cancel style dialogs on your form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-confirm-dialog',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'confirm-dialog-for-elementor-pro-form/',
			],

			///  Dynamic Tags ///

			'ext_tag_cryptocurrency' => [
				'name' => '',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\Cryptocurrency',
				'category' => 'CONTENT',
				'extension_type' => 'dynamic-tag',
				'title' => esc_html__( 'Dynamic Tag - Cryptocurrency', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'The perfect way to use cryptocurrency on all widgets', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-crypto',
				'plugin_depends' => '',
				'doc_url' => 'dynamic-tag-cryptocurrency',
			],
			'ext_tag_posts' => [
				'name' => 'ext_tag_posts',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\Posts',
				'category' => 'POST',
				'extension_type' => 'dynamic-tag',
				'title' => esc_html__( 'Dynamic Tag - Posts', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add support to show posts in Dynamic Tag for text settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-posts',
				'plugin_depends' => '',
				'doc_url' => '',
			],
			'ext_tag_woo_products' => [
				'name' => 'ext_tag_woo_products',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\WooProducts',
				'category' => 'WOO',
				'extension_type' => 'dynamic-tag',
				'title' => esc_html__( 'Dynamic Tag - Products', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add support to show WooCommerce products in Dynamic Tag for text settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-woo-products',
				'plugin_depends' => [ 'woocommerce' ],
				'doc_url' => '',
			],
			'ext_tag_sticky_posts' => [
				'name' => 'ext_tag_sticky_posts',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\StickyPosts',
				'category' => 'POST',
				'extension_type' => 'dynamic-tag',
				'title' => esc_html__( 'Dynamic Tag - Sticky Posts', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add support to show Sticky Posts in Dynamic Tag for text settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-sticky-posts',
				'plugin_depends' => '',
				'doc_url' => '',
			],
			'ext_tag_my_posts' => [
				'name' => 'ext_tag_my_posts',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\MyPosts',
				'category' => 'POST',
				'extension_type' => 'dynamic-tag',
				'title' => esc_html__( 'Dynamic Tag - Posts by the Current User', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add support to show posts of the current user in Dynamic Tag for text settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-my-posts',
				'plugin_depends' => '',
				'doc_url' => '',
			],
			'ext_tag_favorites' => [
				'name' => '',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\Favorites',
				'category' => 'POST',
				'extension_type' => 'dynamic-tag',
				'title' => esc_html__( 'Dynamic Tag - Favorites', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add support for Favorites in Dynamic Tag for text settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-favorites',
				'plugin_depends' => '',
				'doc_url' => 'dynamic-tag-favorites',
			],
			'ext_tag_wishlist' => [
				'name' => '',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\WooWishlist',
				'category' => 'WOO',
				'extension_type' => 'dynamic-tag',
				'title' => esc_html__( 'Dynamic Tag - Woo Wishlist', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add support for Woo Wishlist in Dynamic Tag for text settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-woo-wishlist',
				'plugin_depends' => [ 'woocommerce' ],
				'doc_url' => 'dynamic-tag-woo-wishlist',
			],
			'ext_tag_acf_relationship' => [
				'name' => 'ext_tag_acf_relationship',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\AcfRelationship',
				'category' => 'ACF',
				'extension_type' => 'dynamic-tag',
				'title' => esc_html__( 'Dynamic Tag - ACF Relationship', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add support to show ACF Relationships in Dynamic Tag for text settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-acf-relationship',
				'plugin_depends' => [ 'acf' ],
				'doc_url' => 'dynamic-tag-acf-relationship',
			],
			'ext_tag_metabox_relationship' => [
				'name' => 'ext_tag_metabox_relationship',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\MetaboxRelationship',
				'category' => 'METABOX',
				'extension_type' => 'dynamic-tag',
				'title' => esc_html__( 'Dynamic Tag - Meta Box Relationship', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add support to show Meta Box Relationships in Dynamic Tag for text settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-metabox-relationship',
				'plugin_depends' => [ 'meta-box' ],
				'doc_url' => 'dynamic-tag-meta-box-relationship',
			],
			'ext_tag_terms' => [
				'name' => 'ext_tag_terms',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\Terms',
				'category' => 'TERMS',
				'extension_type' => 'dynamic-tag',
				'title' => esc_html__( 'Dynamic Tag - Terms', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add support to show terms in Dynamic Tag for text settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-terms',
				'plugin_depends' => '',
				'doc_url' => 'dynamic-tag-terms',
			],
			'ext_tag_tags' => [
				'name' => 'ext_tag_tags',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\Tags',
				'category' => 'TERMS',
				'extension_type' => 'dynamic-tag',
				'title' => esc_html__( 'Dynamic Tag - Tags', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add support to show tags in Dynamic Tag for text settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-tags',
				'plugin_depends' => '',
				'doc_url' => 'dynamic-tag-tags',
			],
			'ext_tag_woo_product_categories' => [
				'name' => 'ext_tag_woo_product_categories',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\WooProductCategories',
				'category' => 'WOO',
				'extension_type' => 'dynamic-tag',
				'title' => esc_html__( 'Dynamic Tag - Woo Product Categories', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add support to show WooCommerce Product Categories in Dynamic Tag for text settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-woo-product-categories',
				'plugin_depends' => [ 'woocommerce' ],
				'doc_url' => 'dynamic-tag-woo-product-categories',
			],
			'ext_template' => [
				'name' => 'dce_extension_template',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\Template',
				'category' => 'CONTENT',
				'extension_type' => 'dynamic-tag',
				'title' => esc_html__( 'Dynamic Tag - Template', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add support for Template in Dynamic Tag for text, HTML and textarea settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-template',
				'plugin_depends' => '',
				'doc_url' => 'dynamic-tag-template',
			],
			'ext_token' => [
				'name' => 'dce_extension_token',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\Token',
				'category' => 'CONTENT',
				'extension_type' => 'dynamic-tag',
				'title' => esc_html__( 'Dynamic Tag - Tokens', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add support for Tokens in Dynamic Tag for all settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-tokens',
				'plugin_depends' => '',
				'doc_url' => 'dynamic-tag-token/',
			],
			'dynamic_tag_php' => [
				'name' => 'dce_dynamic_tag_php',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\Php',
				'category' => 'DEV',
				'extension_type' => 'dynamic-tag',
				'title' => esc_html__( 'Dynamic Tag - PHP', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add support for PHP Code in Dynamic Tag for all settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-php',
				'plugin_depends' => '',
				'doc_url' => 'dynamic-tag-php/',
			],
			'dynamic_tag_author_field' => [
				'name' => 'ext_tag_author_field',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\AuthorField',
				'category' => 'USER',
				'extension_type' => 'dynamic-tag',
				'title' => esc_html__( 'Dynamic Tag - Author Field', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add support to show author user fields for text settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-user-field',
				'plugin_depends' => '',
				'doc_url' => 'dynamic-tag-author-field/',
			],
			'dynamic_tag_user_field' => [
				'name' => 'ext_tag_user_field',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\UserField',
				'category' => 'USER',
				'extension_type' => 'dynamic-tag',
				'title' => esc_html__( 'Dynamic Tag - User Field', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add support to show user fields for text settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-user-field',
				'plugin_depends' => '',
				'doc_url' => 'dynamic-tag-user-field/',
			],
			'dynamic_tag_image_token' => [
				'name' => 'dce_dynamic_image_token',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\ImageToken',
				'category' => 'CONTENT',
				'extension_type' => 'dynamic-tag',
				'title' => esc_html__( 'Dynamic Tag - Image Token', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'A Dynamic Tag for Tokens that return Images and Media content', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-tokens',
				'plugin_depends' => '',
				'doc_url' => 'dynamic-tag-image-token/',
			],

			/// Widgets ///

			'wdg_acf' => [
				'class' => 'Widgets\\AcfFields',
				'type' => 'widget',
				'category' => 'ACF',
				'name' => 'dyncontel-acf',
				'title' => esc_html__( 'ACF Fields', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add a custom field created with Advanced Custom Fields', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-acf-fields',
				'plugin_depends' => [ 'acf' ],
				'doc_url' => 'acf-fields/',
				'keywords' => [ 'Advanced Custom Fields' ],
			],
			'wdg_gallery' => [
				'class' => 'Widgets\\AcfGallery',
				'type' => 'widget',
				'category' => 'ACF',
				'name' => 'dyncontel-acfgallery',
				'title' => esc_html__( 'ACF Gallery', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Use images from an ACF Gallery field and display them in a variety of formats', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-grid',
				'plugin_depends' => [ 'advanced-custom-fields-pro' ],
				'doc_url' => 'acf-gallery/',
				'keywords' => [ 'Advanced Custom Fields', 'gallery', 'fields', 'images', 'image' ],
			],
			'wdg_relationship' => [
				'class' => 'Widgets\\AcfRelationshipOldVersion',
				'type' => 'widget',
				'category' => 'ACF',
				'legacy' => true,
				'replaced_by' => 'wdg_acf_relationship',
				'tag' => [ 'loop' ],
				'name' => 'dyncontel-acf-relation',
				'title' => esc_html__( 'ACF Relationship (old version)', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Use the ACF Relationship field to easily display related content', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-relation',
				'plugin_depends' => [ 'acf' ],
				'doc_url' => 'acf-relationship-old-version/',
				'keywords' => [ 'Advanced Custom Fields', 'fields' ],
			],
			'wdg_acf_relationship' => [
				'class' => 'Widgets\\AcfRelationship',
				'type' => 'widget',
				'category' => 'ACF',
				'name' => 'dce-acf-relationship',
				'title' => esc_html__( 'ACF Relationship', 'dynamic-content-for-elementor' ),
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'description' => esc_html__( 'Use the ACF Relationship field to easily display related content', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-relation',
				'plugin_depends' => [ 'acf' ],
				'doc_url' => 'acf-relationship-new-version/',
				'keywords' => [ 'Advanced Custom Fields', 'fields' ],
			],
			'wdg_repeater' => [
				'class' => 'Widgets\\AcfRepeaterOldVersion',
				'type' => 'widget',
				'category' => 'ACF',
				'legacy' => true,
				'replaced_by' => 'wdg_acf_repeater_v2',
				'tag' => [ 'loop' ],
				'name' => 'dyncontel-acf-repeater',
				'title' => esc_html__( 'ACF Repeater (old version)', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Take advantage of the power and flexibility of the ACF Repeater field', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-repeater',
				'plugin_depends' => [ 'advanced-custom-fields-pro' ],
				'doc_url' => 'acf-repeater-fields/',
				'keywords' => [ 'Advanced Custom Fields' ],
			],
			'wdg_acf_repeater_v2' => [
				'class' => 'Widgets\\AcfRepeater',
				'type' => 'widget',
				'category' => 'ACF',
				'tag' => [ 'loop' ],
				'name' => 'dce-acf-repeater-v2',
				'title' => esc_html__( 'ACF Repeater', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Take advantage of the power and flexibility of the ACF Repeater field', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-repeater',
				'plugin_depends' => [ 'advanced-custom-fields-pro' ],
				'doc_url' => 'acf-repeater-fields/',
				'keywords' => [ 'Advanced Custom Fields' ],
			],
			'wdg_acf_flexible_content' => [
				'class' => 'Widgets\\AcfFlexibleContent',
				'type' => 'widget',
				'category' => 'ACF',
				'tag' => [ 'loop' ],
				'name' => 'dce-flexible-content',
				'title' => esc_html__( 'ACF Flexible Content', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Insert a Flexible Content field using layouts and sub fields', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-flexible-content',
				'plugin_depends' => [ 'advanced-custom-fields-pro' ],
				'doc_url' => 'acf-flexible-content/',
				'keywords' => [ 'Advanced Custom Fields' ],
			],
			'wdg_acf_frontend' => [
				'class' => 'Widgets\\AcfFrontend',
				'type' => 'widget',
				'category' => 'ACF',
				'admin_only' => true,
				'name' => 'dce-acf-frontend',
				'title' => esc_html__( 'ACF Frontend Form', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Create a form on the frontend to edit ACF Fields', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-frontend-form',
				'plugin_depends' => [ 'acf' ],
				'doc_url' => 'acf-frontend-form/',
				'keywords' => [ 'Advanced Custom Fields' ],
			],
			'wdg_slider' => [
				'class' => 'Widgets\\AcfSlider',
				'type' => 'widget',
				'category' => 'ACF',
				'name' => 'dyncontel-acfslider',
				'title' => esc_html__( 'ACF Slider', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Use images from an ACF Gallery field to create an image carousel', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-slider',
				'plugin_depends' => [ 'advanced-custom-fields-pro' ],
				'doc_url' => 'acf-slider/',
				'keywords' => [ 'Advanced Custom Fields', 'gallery', 'images', 'carousel' ],
			],
			'wdg_barcode' => [
				'class' => 'Widgets\\QrAndBarcodes',
				'type' => 'widget',
				'category' => 'CONTENT',
				'name' => 'dce_barcode',
				'title' => esc_html__( 'QR & Barcodes', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Quick creation for 1D and 2D barcodes, like EAN and QR Codes', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-qrcode',
				'plugin_depends' => '',
				'doc_url' => 'qr-and-bars-code/',
				'keywords' => [ 'barcode', 'qr' ],
			],
			'wdg_calendar' => [
				'class' => 'Widgets\\AddToCalendar',
				'type' => 'widget',
				'category' => 'CONTENT',
				'name' => 'dce_add_to_calendar',
				'title' => esc_html__( 'Add to Calendar', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add an event to Google Calendar, iOS, Outlook, Outlook.com or Yahoo', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-add-calendar',
				'plugin_depends' => '',
				'doc_url' => 'button-calendar/',
				'keywords' => [ 'date', 'calendar', 'ics', 'reminder' ],
			],
			'wdg_clipboard' => [
				'class' => 'Widgets\\CopyToClipboard',
				'type' => 'widget',
				'category' => 'CONTENT',
				'name' => 'dce-copy-to-clipboard',
				'title' => esc_html__( 'Copy to Clipboard', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Copy the specified text to the User Clipboard', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-button-copy',
				'plugin_depends' => '',
				'doc_url' => 'button-copy-to-clipboard/',
			],
			'wdg_dyncookie' => [
				'class' => 'Widgets\\DynamicCookie',
				'type' => 'widget',
				'category' => 'CONTENT',
				'admin_only' => true,
				'name' => 'dce-dynamiccookie',
				'title' => esc_html__( 'Dynamic Cookie', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Create and manage cookies using dynamic content', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-cookie',
				'plugin_depends' => '',
				'doc_url' => 'dynamic-cookie',
			],
			'wdg_favorites' => [
				'class' => 'Widgets\\AddToFavorites',
				'type' => 'widget',
				'category' => 'FAVORITES',
				'name' => 'dce-add-to-favorites',
				'title' => esc_html__( 'Add to Favorites', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Create a favorite list for your users', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-like',
				'plugin_depends' => '',
				'doc_url' => 'add-to-favorites/',
				'keywords' => [ 'favourites' ],
			],
			'wdg_wishlist' => [
				'class' => 'Widgets\\AddToWooWishlist',
				'type' => 'widget',
				'category' => 'FAVORITES',
				'name' => 'dce-add-to-woo-wishlist',
				'title' => esc_html__( 'Add to Woo Wishlist', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Quickly and easily add WooCommerce products to the wishlist', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-woo-wishlist',
				'plugin_depends' => [ 'woocommerce' ],
				'doc_url' => 'woocommerce-woo-add-to-wishlist/',
				'keywords' => [ 'woocommerce', 'favourites', 'favorites', 'wishlist' ],
			],
			'wdg_modal_window' => [
				'class' => 'Widgets\\FireModalWindow',
				'type' => 'widget',
				'category' => 'CONTENT',
				'name' => 'dyncontel-modalwindow',
				'title' => esc_html__( 'Fire Modal Window', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add a Modal Window button to your pages to create a popup window', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-modal',
				'plugin_depends' => '',
				'doc_url' => 'fire-modalwindow/',
				'keywords' => [ 'popup', 'pop-up', 'modal' ],
			],
			'wdg_pdf' => [
				'class' => 'Widgets\\PdfButton',
				'type' => 'widget',
				'category' => 'PDF',
				'admin_only' => true,
				'name' => 'dce_pdf_button',
				'title' => esc_html__( 'PDF Button', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Dynamically generate a PDF document using your content and styles', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-button-pdf',
				'plugin_depends' => '',
				'doc_url' => 'pdf-button/',
				'keywords' => [ 'generate pdf', 'pdfs' ],
			],
			'wdg_popup' => [
				'class' => 'Widgets\\Modals',
				'type' => 'widget',
				'category' => 'CONTENT',
				'name' => 'dyncontel-popup',
				'title' => esc_html__( 'Modals', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add a modal window to your page', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-popups',
				'plugin_depends' => '',
				'doc_url' => 'popups/',
				'keywords' => [ 'popup', 'pop-up' ],
			],
			'wdg_charts' => [
				'class' => 'Widgets\\DynamicCharts',
				'type' => 'widget',
				'category' => 'CONTENT',
				'name' => 'dce-dynamic-charts',
				'title' => esc_html__( 'Dynamic Charts', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Create beautiful charts in no time at all. Plus, you can easily insert data into your charts or use a CSV file', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-charts',
				'plugin_depends' => '',
				'doc_url' => 'dynamic-charts/',
				'keywords' => [ 'chart', 'radar', 'doughtnut', 'pie', 'line', 'bar' ],
			],
			'wdg_template' => [
				'class' => 'Widgets\\DynamicTemplate',
				'type' => 'widget',
				'category' => 'CONTENT',
				'name' => 'dyncontel-template',
				'title' => esc_html__( 'Dynamic Template', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Insert an existing template and dynamically assign its content', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-layout',
				'plugin_depends' => '',
				'doc_url' => 'dynamic-template/',
			],
			'wdg_tokens' => [
				'class' => 'Widgets\\TextEditorWithTokens',
				'type' => 'widget',
				'category' => 'CONTENT',
				'admin_only' => true,
				'name' => 'dce-tokens',
				'title' => esc_html__( 'Text Editor with Tokens', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add Tokens to show values from posts, users, terms, custom fields, options and others', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-tokens',
				'plugin_depends' => '',
				'doc_url' => 'tokens/',
			],
			'wdg_discover_tokens' => [
				'class' => 'Widgets\\DiscoverTokens',
				'type' => 'widget',
				'category' => 'CONTENT',
				'admin_only' => true,
				'legacy' => true,
				'name' => 'dce-discover-tokens',
				'title' => esc_html__( 'Discover Tokens', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Discover a multitude of tokens on your site to better understand its full power', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-discover-tokens',
				'plugin_depends' => '',
				'doc_url' => 'discover-tokens',
				'keywords' => [ 'tokens', 'list', 'examples' ],
			],
			'wdg_text_timber' => [
				'class' => 'Widgets\\TextEditorWithTimber',
				'type' => 'widget',
				'category' => 'CONTENT',
				'name' => 'dce-text-timber',
				'admin_only' => true,
				'title' => esc_html__( 'Text Editor with Timber', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Write using the Twig Template Engine thanks to integration with Timber', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-text-editor-timber',
				'plugin_depends' => [ 'timber' ],
				'doc_url' => 'text-editor-timber-twig-elementor',
				'keywords' => [ 'twig' ],
			],
			'wdg_animate_text' => [
				'class' => 'Widgets\\AnimatedText',
				'type' => 'widget',
				'category' => 'CREATIVE',
				'name' => 'dyncontel-animateText',
				'title' => esc_html__( 'Animated Text', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Advanced animation for your text', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-animated-text',
				'plugin_depends' => '',
				'doc_url' => 'animated-text/',
				'keywords' => [ 'animations' ],
			],
			'wdg_parallax' => [
				'class' => 'Widgets\\Parallax',
				'type' => 'widget',
				'category' => 'CREATIVE',
				'name' => 'dyncontel-parallax',
				'title' => esc_html__( 'Parallax', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Manage parallax movements with your mouse on desktop or device orientation on mobile', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-parallax',
				'plugin_depends' => '',
				'doc_url' => 'parallax/',
			],
			'wdg_threesixty_slider' => [
				'class' => 'Widgets\\ThreesixtySlider',
				'type' => 'widget',
				'category' => 'CREATIVE',
				'admin_only' => true,
				'name' => 'dyncontel-threesixtyslider',
				'title' => esc_html__( '360 Slider', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Generate a rotation through a series of images', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-360',
				'plugin_depends' => '',
				'doc_url' => '360-slider/',
				'keywords' => [ 'images', 'products' ],
			],
			'wdg_tilt' => [
				'class' => 'Widgets\\Tilt',
				'type' => 'widget',
				'category' => 'CREATIVE',
				'name' => 'dyncontel-tilt',
				'title' => esc_html__( 'Tilt', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Parallax hover tilt effect applied to a template', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-tilt',
				'plugin_depends' => '',
				'doc_url' => 'tilt/',
				'keywords' => [ 'parallax' ],
			],
			'wdg_twentytwenty' => [
				'class' => 'Widgets\\BeforeAfter',
				'type' => 'widget',
				'category' => 'CREATIVE',
				'name' => 'dyncontel-twentytwenty',
				'title' => esc_html__( 'Before After', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Display an image with a before - after effect, ideal for comparing differences between two images', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-before-after',
				'plugin_depends' => '',
				'doc_url' => 'before-after/',
				'keywords' => [ 'images', 'image' ],
			],
			'wdg_doshortcode' => [
				'class' => 'Widgets\\ShortcodeWithTokens',
				'type' => 'widget',
				'category' => 'DEV',
				'admin_only' => true,
				'name' => 'dyncontel-doshortcode',
				'title' => esc_html__( 'Shortcode with Tokens', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Apply a WordPress shortcode using Tokens', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-shortcodes-tokens',
				'plugin_depends' => '',
				'doc_url' => 'shortcode-with-tokens/',
				'keywords' => [ 'shortcode', 'tokens', 'token' ],
			],
			'wdg_includefile' => [
				'class' => 'Widgets\\FileInclude',
				'type' => 'widget',
				'category' => 'DEV',
				'admin_only' => true,
				'legacy' => true,
				'name' => 'dyncontel-includefile',
				'title' => esc_html__( 'File Include', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Directly include files from a path in root as if you were writing in a theme', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-file-include',
				'plugin_depends' => '',
				'doc_url' => 'file-include/',
			],
			'wdg_rawphp' => [
				'class' => 'Widgets\\PhpRaw',
				'type' => 'widget',
				'category' => 'DEV',
				'admin_only' => true,
				'name' => 'dce-rawphp',
				'title' => esc_html__( 'PHP Raw', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add PHP code directly in Elementor', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-php-raw',
				'plugin_depends' => '',
				'doc_url' => 'php-raw/',
				'keywords' => [ 'code' ],
			],
			'wdg_remotecontent' => [
				'class' => 'Widgets\\DynamicApi',
				'type' => 'widget',
				'category' => 'DEV',
				'admin_only' => true,
				'name' => 'dyncontel-remotecontent',
				'title' => esc_html__( 'Dynamic API', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Fetch JSON data and use it dynamically', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-remotecontent',
				'plugin_depends' => '',
				'doc_url' => 'dynamic-api/',
				'keywords' => [ 'json', 'api', 'remote content' ],
			],
			'wdg_web_scraper' => [
				'class' => 'Widgets\\WebScraper',
				'type' => 'widget',
				'category' => 'DEV',
				'admin_only' => true,
				'name' => 'dce-web-scraper',
				'title' => esc_html__( 'Web Scraper', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Scrape and extract content from any website in real-time', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-remotecontent',
				'plugin_depends' => '',
				'doc_url' => 'web-scraper/',
				'keywords' => [ 'remote content' ],
			],
			'wdg_iframe' => [
				'class' => 'Widgets\\Iframe',
				'type' => 'widget',
				'category' => 'CONTENT',
				'admin_only' => true,
				'name' => 'dce-iframe',
				'title' => esc_html__( 'iFrame', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Embed content from external URLs using an iFrame', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-iframe',
				'plugin_depends' => '',
				'doc_url' => 'iframe/',
				'keywords' => [ 'i-frame', 'remote content' ],
			],
			'wdg_lazy_iframe' => [
				'class' => 'Widgets\\LazyIframe',
				'type' => 'widget',
				'category' => 'CONTENT',
				'admin_only' => true,
				'name' => 'dce-lazy-iframe',
				'title' => esc_html__( 'Lazy iFrame', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Embed content from external URLs using an iFrame with lazy loading', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-iframe',
				'plugin_depends' => '',
				'doc_url' => 'lazy-iframe/',
				'keywords' => [ 'i-frame' ],
			],
			'wdg_google_doc_preview' => [
				'class' => 'Widgets\\DocumentViewer',
				'type' => 'widget',
				'category' => 'CONTENT',
				'admin_only' => true,
				'name' => 'dce-google-document-preview',
				'title' => esc_html__( 'Document Viewer', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Display a preview of different document types', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-google-document-preview',
				'plugin_depends' => '',
				'doc_url' => 'document-viewer/',
				'keywords' => [ 'document, preview, docs', 'remote content' ],
			],
			'wdg_dynamicposts' => [
				'class' => 'Widgets\\DynamicPostsOldVersion',
				'type' => 'widget',
				'category' => 'LIST',
				'legacy' => true,
				'replaced_by' => 'wdg_dynamicposts_v2',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dyncontel-acfposts',
				'title' => esc_html__( 'Dynamic Posts (old version)', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Create archives from lists of articles with different query source options. You can display the list with different skins and use templates to control item content', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-posts',
				'plugin_depends' => '',
				'doc_url' => 'dynamic-posts/',
				'keywords' => [ 'list', 'articles', 'carousel', 'slider', 'timeline', 'archive' ],
			],
			'wdg_dynamicposts_v2' => [
				
				'class' => 'Widgets\\DynamicPosts',
				'type' => 'widget',
				'category' => 'LIST',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-dynamicposts-v2',
				'title' => esc_html__( 'Dynamic Posts', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Display lists of posts with different query source options. You can showcase the list using various skins and choose which items to display', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-posts',
				'plugin_depends' => '',
				'doc_url' => 'dynamic-posts-v2',
				'keywords' => [ 'grid', '3d', 'skin', 'list', 'articles', 'carousel', 'slider', 'timeline', 'archive', 'custom', 'dynamic posts v2', 'custom post type', 'cpt', 'item', 'loop', 'v2', 'pods', 'acf', 'fields' ],
			],
			'wdg_dynamic_woo_products' => [
				
				'class' => 'Widgets\\DynamicWooProducts',
				'type' => 'widget',
				'category' => 'WOO',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-dynamic-woo-products',
				'title' => esc_html__( 'Dynamic Products', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Display your WooCommerce products with different skins and choose the items you want to display', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-woo-products',
				'plugin_depends' => [ 'woocommerce' ],
				'doc_url' => 'dynamic-products/',
				'keywords' => [ 'woocommerce', 'woo', 'product', 'list', 'carousel', 'slider', 'timeline', 'loop' ],
			],
			'wdg_dynamic_woo_products_on_sale' => [
				
				'class' => 'Widgets\\DynamicWooProductsOnSale',
				'type' => 'widget',
				'category' => 'WOO',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-dynamic-woo-products-on-sale',
				'title' => esc_html__( 'Dynamic Products On Sale', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Display your WooCommerce products on sale with different skins and choose the items you want to display', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-woo-products-on-sale',
				'plugin_depends' => [ 'woocommerce' ],
				'doc_url' => 'dynamic-products-on-sale/',
				'keywords' => [ 'woocommerce', 'woo', 'product', 'list', 'carousel', 'slider', 'timeline', 'loop', 'sale' ],
			],
			'wdg_woo_products_cart' => [
				
				'class' => 'Widgets\\WooProductsCart',
				'type' => 'widget',
				'category' => 'WOO',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-woo-products-cart',
				'title' => esc_html__( 'Products in the Cart', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Show your customers the WooCommerce products they have in their cart', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-woo-products-cart',
				'plugin_depends' => [ 'woocommerce' ],
				'doc_url' => 'products-in-the-cart/',
				'keywords' => [ 'woocommerce', 'woo', 'product', 'list', 'carousel', 'slider', 'basket', 'loop' ],
			],
			'wdg_woo_product_upsells' => [
				
				'class' => 'Widgets\\WooProductUpsells',
				'type' => 'widget',
				'category' => 'WOO',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-woo-product-upsells',
				'title' => esc_html__( 'Product Upsells', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Show upsells products for a WooCommerce product', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-woo-product-upsells',
				'plugin_depends' => [ 'woocommerce' ],
				'doc_url' => 'product-upsells/',
				'keywords' => [ 'woocommerce', 'woo', 'products', 'list', 'carousel', 'slider', 'loop' ],
			],
			'wdg_woo_product_variations' => [
				
				'class' => 'Widgets\\WooProductVariations',
				'type' => 'widget',
				'category' => 'WOO',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-woo-products-variations',
				'title' => esc_html__( 'Product Variations', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Show variations for a WooCommerce product', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-woo-product-variations',
				'plugin_depends' => [ 'woocommerce' ],
				'doc_url' => 'woo-product-variations/',
				'keywords' => [ 'woocommerce', 'woo', 'products', 'variations', 'list', 'carousel', 'slider', 'loop' ],
			],
			'wdg_woo_product_crosssells' => [
				
				'class' => 'Widgets\\WooProductCrosssells',
				'type' => 'widget',
				'category' => 'WOO',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-woo-product-crosssells',
				'title' => esc_html__( 'Product Cross-Sells', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Show cross-sells products for a WooCommerce product', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-woo-product-crosssells',
				'plugin_depends' => [ 'woocommerce' ],
				'doc_url' => 'product-cross-sells/',
				'keywords' => [ 'woocommerce', 'woo', 'products', 'related', 'list', 'carousel', 'slider', 'basket', 'loop' ],
			],
			'wdg_show_favorites' => [
				
				'class' => 'Widgets\\ShowFavorites',
				'type' => 'widget',
				'category' => 'FAVORITES',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-dynamic-show-favorites',
				'icon' => 'icon-dce-show-favorites',
				'title' => esc_html__( 'Favorites', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Show your customers the favorites they have. Works in combination with the "Add to Favorites" widget', 'dynamic-content-for-elementor' ),
				'plugin_depends' => '',
				'doc_url' => 'favorites/',
				'keywords' => [ 'favourites', 'list', 'carousel', 'slider', 'loop', 'add to favorites' ],
			],
			'wdg_show_wishlist' => [
				
				'class' => 'Widgets\\WooWishlist',
				'type' => 'widget',
				'category' => 'FAVORITES',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-woo-wishlist',
				'icon' => 'icon-dce-show-wishlist',
				'title' => esc_html__( 'Woo Wishlist', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Show your customers their wishlist automatically, excluding bought products. Works in combination with the "Add to Woo Wishlist" widget', 'dynamic-content-for-elementor' ),
				'plugin_depends' => [ 'woocommerce' ],
				'doc_url' => 'woocommerce-woo-wishlist/',
				'keywords' => [ 'woocommerce', 'woo', 'favourites', 'product', 'list', 'carousel', 'slider', 'basket', 'loop' ],
			],
			'wdg_stick_yposts' => [
				
				'class' => 'Widgets\\StickyPosts',
				'type' => 'widget',
				'category' => 'LIST',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-sticky-posts',
				'title' => esc_html__( 'Sticky Posts', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Display Sticky Posts with different skins and choose the items you want to display', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-sticky-posts',
				'plugin_depends' => '',
				'doc_url' => 'sticky-posts/',
				'keywords' => [ 'grid', '3d', 'skin', 'list', 'articles', 'carousel', 'slider' ],
			],
			'wdg_search_results' => [
				
				'class' => 'Widgets\\SearchResults',
				'type' => 'widget',
				'category' => 'LIST',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-search-results',
				'title' => esc_html__( 'Search Results', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Display Search Results with different skins and choose the items you want to display', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-search-results',
				'plugin_depends' => '',
				'doc_url' => 'search-results/',
				'keywords' => [ 'grid', '3d', 'skin', 'list', 'articles', 'carousel', 'slider', 'searching' ],
			],
			'wdg_my_posts' => [
				
				'class' => 'Widgets\\MyPosts',
				'type' => 'widget',
				'category' => 'LIST',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-my-posts',
				'title' => esc_html__( 'Posts by the Current User', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Show your logged-in customers posts written by them', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-my-posts',
				'plugin_depends' => '',
				'doc_url' => 'my-posts/',
				'keywords' => [ 'grid', '3d', 'skin', 'list', 'articles', 'carousel', 'slider', 'searching', 'current author', 'author' ],
			],
			'wdg_dynamic_archives' => [
				
				'class' => 'Widgets\\DynamicArchives',
				'type' => 'widget',
				'category' => 'LIST',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-dynamic-archives',
				'title' => esc_html__( 'Dynamic Archives', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Displays posts for archives and categories, offering a tailored way to showcase and manage items', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-archives',
				'plugin_depends' => '',
				'doc_url' => 'dynamic-archives/',
				'keywords' => [ 'grid', '3d', 'skin', 'list', 'articles', 'carousel', 'slider' ],
			],
			'wdg_metabox_relationship' => [
				
				'class' => 'Widgets\\MetaboxRelationship',
				'type' => 'widget',
				'category' => 'METABOX',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-metabox-relationship',
				'title' => esc_html__( 'Meta Box Relationship', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Use Meta Box Relationships to easily display related content', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-metabox-relationship',
				'plugin_depends' => [ 'meta-box' ],
				'doc_url' => 'meta-box-relationship-elementor/',
				'keywords' => [ 'metabox' ],
			],
			'wdg_google_maps' => [
				'class' => 'Widgets\\DynamicGoogleMaps',
				'type' => 'widget',
				'category' => 'MAPS',
				'admin_only' => true,
				'name' => 'dyncontel-acf-google-maps',
				'title' => esc_html__( 'Dynamic Google Maps', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Build a map using data from ACF Google Map fields, Meta Box Google Map fields, addresses or latitude and longitude', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-map',
				'plugin_depends' => '',
				'doc_url' => 'dynamic-google-maps/',
				'keywords' => [ 'Advanced Custom Fields', 'fields' ],
			],
			'wdg_google_maps_directions' => [
				'class' => 'Widgets\\DynamicGoogleMapsDirections',
				'type' => 'widget',
				'category' => 'MAPS',
				'name' => 'dce-dynamic-google-maps-directions',
				'title' => esc_html__( 'Dynamic Google Maps Directions', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Show the map and calculate the direction between two locations. Can be used in conjunction with its Dynamic Tags to show distance info and directions', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-google-maps-directions',
				'plugin_depends' => '',
				'doc_url' => 'dynamic-google-maps-directions',
				'keywords' => [ 'distance' ],
			],
			'wdg_metabox_google_maps' => [
				'class' => 'Widgets\\MetaBoxGoogleMaps',
				'type' => 'widget',
				'category' => 'METABOX',
				'name' => 'dce-metabox-google-maps',
				'title' => esc_html__( 'Meta Box Google Maps', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Build a map using data from Meta Box Google Map fields', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-metabox-google-maps',
				'plugin_depends' => [ 'meta-box' ],
				'doc_url' => 'meta-box-google-maps-elementor/',
				'keywords' => [ 'metabox', 'fields' ],
			],
			'wdg_dynamic_users' => [
				'class' => 'Widgets\\DynamicUsers',
				'type' => 'widget',
				'category' => 'LIST',
				'admin_only' => true,
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dyncontel-dynamicusers',
				'title' => esc_html__( 'Dynamic Users', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Create an archive based on User profiles', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-users',
				'plugin_depends' => '',
				'doc_url' => 'dynamic-users/',
				'keywords' => [ 'list', 'users', 'archive' ],
			],
			'wdg_animated_offcanvas_menu' => [
				'class' => 'Widgets\\AnimatedOffCanvasMenu',
				'type' => 'widget',
				'category' => 'CREATIVE',
				'name' => 'dce-animatedoffcanvasmenu',
				'title' => esc_html__( 'Animated Off-Canvas Menu', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'An off-canvas menu is positioned outside of the viewport and slides in when activated', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-animatedoffcanvasmenu',
				'plugin_depends' => '',
				'doc_url' => 'animated-offcanvas-menu/',
				'keywords' => [ 'menu', 'animations' ],
			],
			'wdg_cursor_tracker' => [
				'class' => 'Widgets\\CursorTracker',
				'type' => 'widget',
				'category' => 'CREATIVE',
				'name' => 'dyncontel-cursorTracker',
				'title' => esc_html__( 'Cursor Tracker', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'An element that follows the cursor and indicates the level of scrolling', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-cursor-tracker',
				'plugin_depends' => '',
				'doc_url' => 'cursor-tracker/',
			],
			'wdg_file_browser' => [
				'class' => 'Widgets\\FileBrowser',
				'type' => 'widget',
				'category' => 'LIST',
				'default_status' => 'inactive',
				'admin_only' => true,
				'name' => 'dce-filebrowser',
				'title' => esc_html__( 'FileBrowser', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Display a list of files you uploaded. This is particularly useful when you need to make pictures or documents available in a simple and intuitive way', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-filebrowser',
				'plugin_depends' => '',
				'doc_url' => 'file-browser/',
				'keywords' => [ 'list', 'file', 'download', 'archive' ],
			],
			'wdg_parent_child_menu' => [
				'class' => 'Widgets\\ParentChildMenu',
				'type' => 'widget',
				'category' => 'LIST',
				'name' => 'parent-child-menu',
				'title' => esc_html__( 'Parent Child Menu', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Build a list of entries in horizontal or vertical mode where a parent element can be considered as a menu', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-parentchild',
				'plugin_depends' => '',
				'doc_url' => 'widget-parent-child-menu/',
			],
			'wdg_search_filter' => [
				'class' => 'Widgets\\AdvancedFilteringSearchAndFilterPro',
				'type' => 'widget',
				'category' => 'LIST',
				'name' => 'dce-searchfilter',
				'title' => esc_html__( 'Advanced Filtering from Search & Filter Pro', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'The Ultimate WordPress filter plugin that uses Ajax with Elementor', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-search-filter',
				'plugin_depends' => [ 'search-filter-pro' ],
				'doc_url' => 'search-filter-pro-elementor',
				'keywords' => [ 'filters', 'Search and Filter Pro' ],
			],
			'wdg_single_posts_menu' => [
				'class' => 'Widgets\\SinglePostsList',
				'type' => 'widget',
				'category' => 'LIST',
				'name' => 'single-posts-menu',
				'title' => esc_html__( 'Single Posts List', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Create a custom menu from single pages', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-listsingle',
				'plugin_depends' => '',
				'doc_url' => 'single-posts-list/',
			],
			'wdg_taxonomy_terms_menu' => [
				'class' => 'Widgets\\TaxonomyTermsList',
				'type' => 'widget',
				'category' => 'LIST',
				'name' => 'taxonomy-terms-menu',
				'title' => esc_html__( 'Taxonomy Terms List', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Create an archive type navigation menu based on a taxonomy. It generates a list of the terms that belong to a specific taxonomy so that each item can be clicked to easily access to the term archive', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-parenttax',
				'plugin_depends' => '',
				'doc_url' => 'taxonomy-terms-list/',
				'keywords' => [ 'list', 'menu' ],
			],
			'wdg_views' => [
				'class' => 'Widgets\\Views',
				'type' => 'widget',
				'category' => 'LIST',
				'admin_only' => true,
				'name' => 'dce-views',
				'tag' => [ 'loop' ],
				'title' => esc_html__( 'Views', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Create a custom list from query results', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-views',
				'plugin_depends' => '',
				'doc_url' => 'views/',
				'keywords' => [ 'list', 'articles', 'search', 'archive', 'cpt' ],
			],
			'wdg_pods' => [
				'class' => 'Widgets\\PodsFields',
				'type' => 'widget',
				'category' => 'PODS',
				'name' => 'dyncontel-pods',
				'title' => esc_html__( 'Pods Fields', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add a custom field created with Pods', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-acf-fields',
				'plugin_depends' => [ 'pods' ],
				'doc_url' => 'pods-fields/',
			],
			'wdg_pods_gallery' => [
				'class' => 'Widgets\\PodsGallery',
				'type' => 'widget',
				'category' => 'PODS',
				'name' => 'dyncontel-podsgallery',
				'title' => esc_html__( 'Pods Gallery', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Use a list of images from a Pods Image field with various display options', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-grid',
				'plugin_depends' => [ 'pods' ],
				'doc_url' => 'pods-gallery/',
				'keywords' => [ 'images', 'fields' ],
			],
			'wdg_pods_relationship' => [
				'class' => 'Widgets\\PodsRelationship',
				'type' => 'widget',
				'category' => 'PODS',
				'admin_only' => true,
				'tag' => [ 'loop' ],
				'name' => 'dyncontel-pods-relation',
				'title' => esc_html__( 'Pods Relationship', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Display related posts linked using the Pods Relationship field', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-relation',
				'plugin_depends' => [ 'pods' ],
				'doc_url' => 'pods-relationship/',
			],
			'wdg_breadcrumbs' => [
				'class' => 'Widgets\\Breadcrumbs',
				'type' => 'widget',
				'category' => 'POST',
				'name' => 'dce-breadcrumbs',
				'title' => esc_html__( 'Breadcrumbs', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Insert breadcrumbs and generate paths inside your page automatically', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-breadcrumbs',
				'plugin_depends' => '',
				'doc_url' => 'breadcrumbs/',
			],
			'wdg_content' => [
				'class' => 'Widgets\\Content',
				'type' => 'widget',
				'category' => 'POST',
				'name' => 'dyncontel-content',
				'title' => esc_html__( 'Content', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Retrieve the post content', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-content',
				'plugin_depends' => '',
				'doc_url' => 'content/',
			],
			'wdg_date' => [
				'class' => 'Widgets\\Date',
				'type' => 'widget',
				'category' => 'POST',
				'name' => 'dyncontel-date',
				'title' => esc_html__( 'Date', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Display the published or modified date', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-date',
				'plugin_depends' => '',
				'doc_url' => 'date/',
			],
			'wdg_excerpt' => [
				'class' => 'Widgets\\Excerpt',
				'type' => 'widget',
				'category' => 'POST',
				'name' => 'dyncontel-excerpt',
				'title' => esc_html__( 'Excerpt', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Display a short section from the beginning of the content', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-excerpt',
				'plugin_depends' => '',
				'doc_url' => 'excerpt/',
				'keywords' => [ 'content' ],
			],
			'wdg_featured_image' => [
				'class' => 'Widgets\\FeaturedImage',
				'type' => 'widget',
				'category' => 'POST',
				'name' => 'dyncontel-featured-image',
				'title' => esc_html__( 'Featured Image', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Retrieve the featured image', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-featured-image',
				'plugin_depends' => '',
				'doc_url' => 'featured-image/',
			],
			'wdg_icon_format' => [
				'class' => 'Widgets\\IconFormat',
				'type' => 'widget',
				'category' => 'POST',
				'name' => 'dyncontel-iconformat',
				'title' => esc_html__( 'Icon Format', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add an icon for your post format and identify its type', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-formats',
				'plugin_depends' => '',
				'doc_url' => 'icon-format/',
			],
			'wdg_meta' => [
				'class' => 'Widgets\\PostMeta',
				'type' => 'widget',
				'category' => 'POST',
				'admin_only' => true,
				'name' => 'dce-meta',
				'title' => esc_html__( 'Post Meta', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Display any post meta saved in the database', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-post-meta',
				'plugin_depends' => '',
				'doc_url' => 'post-meta/',
				'legacy' => true,
			],
			'wdg_next_prev' => [
				'class' => 'Widgets\\PrevNext',
				'type' => 'widget',
				'category' => 'POST',
				'name' => 'dyncontel-post-nextprev',
				'title' => esc_html__( 'Prev Next', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Display navigation links to adjacent posts based on a category/taxonomy or tag', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-prevnext',
				'plugin_depends' => '',
				'doc_url' => 'prevnext/',
				'keywords' => [ 'previous', 'list' ],
			],
			'wdg_read_more' => [
				'class' => 'Widgets\\ReadMore',
				'type' => 'widget',
				'category' => 'POST',
				'legacy' => true,
				'name' => 'dyncontel-readmore',
				'title' => esc_html__( 'Read More Button', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Add a "Read More" button below your post or on the block in the archive, create a call-to-action.', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-readmore',
				'plugin_depends' => '',
				'doc_url' => 'read-more-button/',
			],
			'wdg_terms' => [
				'class' => 'Widgets\\TermsAndTaxonomy',
				'type' => 'widget',
				'category' => 'POST',
				'name' => 'dyncontel-terms',
				'title' => esc_html__( 'Terms & Taxonomy', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Insert your post taxonomies', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-terms',
				'plugin_depends' => '',
				'doc_url' => 'terms-and-taxonomy/',
				'keywords' => [ 'taxonomies' ],
			],
			'wdg_title' => [
				'class' => 'Widgets\\DynamicTitle',
				'type' => 'widget',
				'category' => 'POST',
				'name' => 'dyncontel-title',
				'legacy' => true,
				'title' => esc_html__( 'Dynamic Title', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'It automatically recognises the object among Posts, Archives and Users and gives you the title', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-title',
				'plugin_depends' => '',
				'doc_url' => 'title/',
			],
			'wdg_title_type' => [
				'class' => 'Widgets\\PostType',
				'type' => 'widget',
				'category' => 'POST',
				'name' => 'dyncontel-titleType',
				'legacy' => true,
				'title' => esc_html__( 'Post Type', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Retrieve the post type', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-title-type',
				'plugin_depends' => '',
				'doc_url' => 'title-type/',
				'keywords' => [ 'cpt', 'Custom Post Type' ],
			],
			'wdg_user' => [
				'class' => 'Widgets\\UserFields',
				'type' => 'widget',
				'category' => 'POST',
				'admin_only' => true,
				'legacy' => true,
				'name' => 'dce-user-fields',
				'title' => esc_html__( 'User Fields', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Display any user field', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-userfields',
				'plugin_depends' => '',
				'doc_url' => 'user-fields/',
			],
			'wdg_title_taxonomy' => [
				'class' => 'Widgets\\TaxonomyTitle',
				'type' => 'widget',
				'category' => 'POST',
				'legacy' => true,
				'name' => 'dyncontel-titleTaxonomy',
				'title' => esc_html__( 'Taxonomy Title', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Use the current Taxonomy term as the Archive title', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-title-taxonomy',
				'plugin_depends' => '',
				'doc_url' => 'title-taxonomy/',
			],
			'wdg_svg_blob' => [
				'class' => 'Widgets\\SvgBlob',
				'type' => 'widget',
				'category' => 'SVG',
				'admin_only' => true,
				'name' => 'dyncontel-svgblob',
				'title' => esc_html__( 'SVG Blob', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Create a shape using an SVG path and make it move!', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-svgblob',
				'plugin_depends' => '',
				'doc_url' => 'svg-blob/',
				'keywords' => [ 'animations' ],
			],
			'wdg_svg_distortion' => [
				'class' => 'Widgets\\SvgDistortion',
				'type' => 'widget',
				'category' => 'SVG',
				'admin_only' => true,
				'name' => 'dyncontel-svgdistortion',
				'title' => esc_html__( 'SVG Distortion', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'The SVG Distortion widget operates by calculating the displacement map based on a source image', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-distortion',
				'plugin_depends' => '',
				'doc_url' => 'svg-distortion/',
				'keywords' => [ 'animations' ],
			],
			'wdg_svg_filter_effects' => [
				'class' => 'Widgets\\SvgFilterEffects',
				'type' => 'widget',
				'category' => 'SVG',
				'admin_only' => true,
				'name' => 'dyncontel-filtereffects',
				'title' => esc_html__( 'SVG Filter Effects', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'The widget operates using primitive SVG filters', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-svg-filter',
				'plugin_depends' => '',
				'doc_url' => 'svg-fe-filtereffects/',
			],
			'wdg_svg_imagemask' => [
				'class' => 'Widgets\\SvgImageMask',
				'type' => 'widget',
				'category' => 'SVG',
				'admin_only' => true,
				'name' => 'dyncontel-svgimagemask',
				'title' => esc_html__( 'SVG Image Mask', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'The SVG Image Mask widget operates through the mask attribute of the SVG', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-svg-mask',
				'plugin_depends' => '',
				'doc_url' => 'svg-mask/',
				'keywords' => [ 'masking', 'images' ],
			],
			'wdg_svg_morphing' => [
				'class' => 'Widgets\\SvgMorphing',
				'type' => 'widget',
				'category' => 'SVG',
				'admin_only' => true,
				'name' => 'dyncontel-svgmorphing',
				'title' => esc_html__( 'SVG Morphing', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'You can transpose SVG paths to morph from one shape to another ', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-svg-morphing',
				'plugin_depends' => '',
				'doc_url' => 'svg-morphing/',
				'keywords' => [ 'animations' ],
			],
			'wdg_svg_path_text' => [
				'class' => 'Widgets\\SvgTextPath',
				'type' => 'widget',
				'category' => 'SVG',
				'admin_only' => true,
				'name' => 'dyncontel-svgpathText',
				'title' => esc_html__( 'SVG Text Path', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Write out text along a source based on an SVG path line', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-svgtextpath',
				'plugin_depends' => '',
				'doc_url' => 'svg-text-path/',
			],
			'wdg_toolset' => [
				'class' => 'Widgets\\ToolsetFields',
				'type' => 'widget',
				'category' => 'TOOLSET',
				'name' => 'dyncontel-toolset',
				'title' => esc_html__( 'Toolset Fields', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Display a Toolset custom field', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-acf-fields',
				'plugin_depends' => [ 'types' ],
				'doc_url' => 'toolset-fields/',
			],
			'wdg_toolset_relationship' => [
				'class' => 'Widgets\\ToolsetRelationship',
				'type' => 'widget',
				'category' => 'TOOLSET',
				'admin_only' => true,
				'tag' => [ 'loop' ],
				'name' => 'dyncontel-toolset-relation',
				'title' => esc_html__( 'Toolset Relationship', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Display related posts defined by Toolset Relationships', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-relation',
				'plugin_depends' => [ 'types' ],
				'doc_url' => 'toolset-relationship/',
				'keywords' => [ 'fields' ],
			],
			'wdg_bg_canvas' => [
				'class' => 'Widgets\\BgCanvas',
				'type' => 'widget',
				'category' => 'WEBGL',
				'name' => 'dyncontel-bgcanvas',
				'title' => esc_html__( 'Background Canvas', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Easily integrate WebGL with Canvas for Shader effects', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-canvas',
				'plugin_depends' => '',
				'doc_url' => 'webgl-threejs-background-canvas/',
				'keywords' => [ 'bg', 'images' ],
			],
			'wdg_distortion_image' => [
				'class' => 'Widgets\\ImagesDistortionHover',
				'type' => 'widget',
				'category' => 'WEBGL',
				'admin_only' => true,
				'name' => 'dyncontel-imagesDistortion',
				'title' => esc_html__( 'Images Distortion Hover', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Special rollover effects based on WebGL, Three.js, and transformation shaders', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-distortion',
				'plugin_depends' => '',
				'doc_url' => 'webgl-threejs-images-distortions-hover/',
				'keywords' => [ 'animations' ],
			],
			'wdg_panorama' => [
				'class' => 'Widgets\\Panorama',
				'type' => 'widget',
				'category' => 'WEBGL',
				'name' => 'dyncontel-panorama',
				'title' => esc_html__( 'Panorama', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Display a 360 degree spherical image through VR mode', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-panorama',
				'plugin_depends' => '',
				'doc_url' => 'panorama/',
				'keywords' => [ 'images' ],
			],
			'wdg_osm_map' => [
				'class' => 'Widgets\\OsmMap',
				'type' => 'widget',
				'category' => 'MAPS',
				'name' => 'dce-osm-map',
				'title' => esc_html__( 'OpenStreetMap', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'A map widget that does not need a key thanks to OpenStreetMap!', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-osm',
				'plugin_depends' => '',
				'doc_url' => 'openstreetmap-elementor/',
				'keywords' => [ 'map', 'open street map', 'google maps', 'osm', 'address' ],
			],
			'wdg_dynamic_osm_map' => [
				'class' => 'Widgets\\DynamicOsmMap',
				'type' => 'widget',
				'category' => 'MAPS',
				'name' => 'dce-dynamic-osm-map',
				'title' => esc_html__( 'Dynamic OpenStreetMap', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Display and customize multiple points of interest on OpenStreetMap', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-openstreetmap',
				'plugin_depends' => '',
				'doc_url' => 'dynamic-openstreetmap/',
				'keywords' => [ 'map', 'open street map', 'google maps', 'osm', 'address' ],
			],
			'wdg_cryptocoin_badge' => [
				'class' => 'Widgets\\CryptocoinBadge',
				'type' => 'widget',
				'category' => 'CONTENT',
				'name' => 'dce-cryptocoin-badge',
				'title' => esc_html__( 'Cryptocoin Badge', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Perfect for showing your customers the realtime value of your favorite cryptocoin', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-crypto-badge',
				'plugin_depends' => '',
				'doc_url' => 'cryptocoin-badge',
				'keywords' => [ 'realtime', 'bitcoin', 'crypto' ],
			],
			'wdg_pdf_viewer' => [
				'class' => 'Widgets\\PdfViewer',
				'type' => 'widget',
				'category' => 'PDF',
				'name' => 'dce-pdf-viewer',
				'title' => esc_html__( 'PDF Viewer', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Show a PDF file', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-pdf-viewer',
				'plugin_depends' => '',
				'doc_url' => 'pdf-viewer/',
				'keywords' => [ 'pdf', 'viewer', 'preview' ],
			],

			// Page Settings

			'pst_scrolling' => [
				'class' => 'PageSettings\\PageScroll',
				'type' => 'page-setting',
				'name' => 'dce_document_scrolling',
				'title' => esc_html__( 'Page Scroll', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Turn sections into scrolling steps and minimize pagination using either scroll, snap, or inertia effects', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-page-scroll',
				'plugin_depends' => '',
				'doc_url' => 'page-scroll-document/',
			],

			// Global Settings

			'gst_smooth_transition' => [
				'class' => 'Includes\\Settings\\SmoothTransition',
				'type' => 'global-setting',
				'name' => 'dce_settings_smoothtransition',
				'title' => esc_html__( 'Smooth Transition', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Provides a smooth animated transition between pages', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-smooth-transition',
				'plugin_depends' => [],
				'doc_url' => 'global-settings-smooth-transition/',
				'legacy' => true,
			],
			'gst_tracker_header' => [
				'class' => 'Includes\\Settings\\TrackerHeader',
				'type' => 'global-setting',
				'name' => 'dce_settings_trackerheader',
				'title' => esc_html__( 'Tracker Header', 'dynamic-content-for-elementor' ),
				'description' => esc_html__( 'Tracker header will fix the header of your site to the top of the page. The header will disappear while scrolling but will reappear with a small swipe upward. This will help to optimize available spaces, especially on mobile devices', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-tracker-header',
				'plugin_depends' => [],
				'doc_url' => 'global-settings-tracker-header/',
				'legacy' => true,
			],
		];
	}
}
